open Interval
let rdim = ref 2
let rhighbound = ref 20.
let rlowbound = ref (-20.)
let rprecx = ref 0.2
let rprecfx = ref 0.001
let rfunc = ref 1
let rdisp = ref false
let rexpr= ref "sum(1,n,(xi^2/4000.))-prod(1,n,cos(xi/sqrt(i)))+1."

let bool_of_string x = x = "true"
module Map_strings =
  Map.Make(struct
    type t = string
    let compare x y = compare x y
  end)

let map file = 
  let m = ref Map_strings.empty
  and inch = open_in file
  and sep = Str.regexp "[ \t=]+" in
  try
    while true do
      let st = input_line inch in
      let sttab = Array.of_list (Str.split sep st) in
      if Array.length sttab >= 2
      then m := Map_strings.add sttab.(0) sttab.(1) !m;
    done;
    failwith "cfg.map: Unreachable"
  with End_of_file -> close_in inch; !m;;

let _ = 
  let map = map "func.cfg" in
  let find_val s = 
    let r = Map_strings.find s map in
    r in
  let find_string v x = try v:=(find_val x) with _ -> ()
  and find_float v x = try v:=float_of_string (find_val x) with _ -> ()
  and find_int v x = try v:=int_of_string (find_val x) with _ -> ()
  and find_bool v x = try v:=bool_of_string (find_val x) with _ -> () in
  find_bool rdisp "disp";
  find_int rdim "dim";
  find_int rfunc "func";
  find_string rexpr "expr";
  find_float rprecx "precx";
  find_float rprecfx "precfx";
  find_float rhighbound "high";
  find_float rlowbound "low";;



let anon_fun d = ();;
let _ = 
  Arg.parse [
    ("-disp",Arg.Unit (fun () -> rdisp:=true),": set disp to true (default false; usable only with dim=2)");
    ("-dim",Arg.Set_int rdim,"dim : set dimension to dim (default 2)");
    ("-func",Arg.Set_int rfunc,"f : set func number to f (default 1, 0 to use value of expr)");
    ("-expr",Arg.Set_string rexpr,"s : set expression to s (default:\"sum(1,n,(xi^2/4000.))-prod(1,n,cos(xi/sqrt(i)))+1.)\"");
    ("-precx",Arg.Set_float rprecx,"precx : set precision for x (default 0.2)");
    ("-precfx",Arg.Set_float rprecfx,"precfx : set precision for fx (default 0.001)");
    ("-high",Arg.Set_float rhighbound,"high : set highbound to high (default 20)");
    ("-low",Arg.Set_float rlowbound,"high : set lowbound to low (default -20)")
  ] anon_fun "Programme d'optimisation AG\n (C) JM Alliot/N Durand/D Gianazza/JB Gotteland (2012)";;

let dim = if !rdim<1 then failwith "dim>=1" else !rdim
let highbound = !rhighbound
let lowbound  = if !rlowbound>=highbound then failwith "lowbound<=highbound" else !rlowbound
let func = !rfunc
let precisionx = !rprecx
let precisionfx = !rprecfx
let disp= !rdisp && (dim=2)
let expr = if func=0 then Expr.eval (Expr.string2expr !rexpr)
  else Expr.eval (Expr.string2expr "sum(1,n,(xi^2/4000.))-prod(1,n,cos(xi/sqrt(i)))+1.")
let expri = if func=0 then Expr.range (Expr.string2expr !rexpr)
  else Expr.range (Expr.string2expr "sum(1,n,(xi^2/4000.))-prod(1,n,cos(xi/sqrt(i)))+1.")

(* Standard Griewank function without rotation matrix, very easy to maximize *)




let delta = highbound -. lowbound
let start_inter = 
  Array.init dim (function i -> {low=lowbound;high=highbound})

let pi = acos (-1.0)
let mrot=   
  Rotate.matrot dim 

(* [0,pi] *)
let micha_x v = 
  let m = 10 in
  let s1 = ref 0.0 in 
  for i= 0 to (Array.length v)-1 do
    s1:= !s1+. (sin v.(i)) *. (sin (v.(i)*.v.(i)*.(float (i+1))/.pi))**(2.0*.(float m));
  done;
  ((float dim)+. !s1);;
let micha_X v =
  let m = 10 in 
  let s1 = ref  {low=0.;high=0.} in
  for i= 0 to (Array.length v)-1 do
    s1:= !s1 +$ ((sin_I v.(i)) *$ (pow_I_i (sin_I ((pow_I_i v.(i) 2)*$.((float (i+1))/.pi))) (2*m)));
  done;
  !s1+$.(float dim);;

(* [-500,500] Fonction sans intérêt*)
let schwefel_x v = 
  let s1 = ref 0.0 in
  for i= 0 to (Array.length v)-1 do
    s1:= !s1+. v.(i)*. (sin (sqrt (abs_float v.(i))));
  done;
  !s1+.((float dim)*.(max (abs_float highbound) (abs_float lowbound)));;
let schwefel_X v = 
  let s1 = ref  {low=0.;high=0.} in
  for i= 0 to (Array.length v)-1 do
    s1:= !s1 +$ (v.(i) *$ (sin_I (sqrt_I (abs_I v.(i))))) ;
  done;
  !s1+$.((float dim)*.(max (abs_float highbound) (abs_float lowbound)));;

(* [-4.12,6.12] *)
let rastrigin_x v = 
  let s1 = ref 0.0 in
  for i= 0 to (Array.length v)-1 do
    s1:= !s1+. 10.0 +. v.(i) *. v.(i) -. 10.0*.(cos (2.*.pi*.v.(i)));
  done;
  1./.(1.+. !s1);;
let rastrigin_X v = 
  let s1 = ref  {low=0.;high=0.} in
  for i= 0 to (Array.length v)-1 do
    s1:= ((!s1 +$. 10.0) +$ (pow_I_i v.(i) 2)) -$ ((cos_I (v.(i) *$.(2.*.pi)))*$.10.0)
  done;
  1./.$(1.+.$ !s1);;

(* [-4.12,6.12] *)
let ackley_x v = 
  let a = 20.0 and b=0.2 and c=2.*.pi in
  let s1 = ref 0.0 and s2 = ref 0. in 
  for i= 0 to (Array.length v)-1 do
    s1:= !s1+. v.(i) *. v.(i);
    s2 := !s2 +. (cos (c*.v.(i)))
  done;
  let sum = -.a*.(exp (-.b*.(sqrt (!s1/.(float dim)))))-.(exp (!s2/.(float dim)))
    +.a+.(exp 1.) in
  1./.(1.+. sum);;
let ackley_X v = 
  let a = 20.0 and b=0.2 and c=2.*.pi in
  let s1 = ref  {low=0.;high=0.} and s2 = ref {low=0.;high=0.} in
  for i= 0 to (Array.length v)-1 do
    s1:= !s1 +$ (pow_I_i v.(i) 2);
    s2 := !s2 +$ (cos_I (v.(i) *$. c))
  done;
  let sum = ((exp_I ((sqrt_I (!s1/$.(float dim)))*$.(-.b)))*$.(-.a) -$ (exp_I (!s2/$.(float dim)))) +$. a +$. (exp 1.) in
  1./.$(1. +.$ sum);;


(* [-400,600] *)
let griewank_x v = 
  let s1 = ref 0.0 and s2 = ref 1.0 and sum=ref 0.0 in
  for i= 0 to (Array.length v)-1 do
    s1:= !s1+. v.(i)*. v.(i);
    s2:= !s2*. (cos (v.(i)/. (sqrt (float (i+1)))))
  done;
  sum := !s1/. 4000.0 -. !s2 +.1.;
  1./.(1.+. !sum)
let griewank_X v = 
  let s1 = ref  zero_I and s2 = ref one_I in
  for i= 0 to (Array.length v)-1 do
    s1:= !s1 +$ (pow_I_i v.(i) 2);
    s2:= !s2 *$ (cos_I (v.(i) /$.(sqrt (float (i+1)))))
  done;
  let sum = ((!s1 /$. 4000.0)  -$ !s2) +$. 1. in
  1./.$ (sum +$. 1.)

    
let helice_x p =
  let n=5. and  k=1.1 in
  let x=p.(0) and y=p.(1) in
  let teta=atan2 y x
  and r= sqrt (x*.x+.y*.y) in
  (1./.(1.+.r*.((cos (n*.r+.teta))+.k)))
let helice_X p =
  let n=5. and  k=1.1 in
  let x=p.(0) and y=p.(1) in
  let teta=atan2_I_I y x
  and r= sqrt_I (x *$ x +$ y *$ y) in
  (1. /.$ (1. +.$ r*$((cos_I (n*.$r+$teta))+$.k)))

let f_x= 
  match func with
    0 -> expr
  | 1 -> griewank_x
  | 2 -> Rotate.rotated_F mrot griewank_x
  | 3 -> ackley_x
  | 4 -> Rotate.rotated_F mrot ackley_x
  | 5 -> rastrigin_x
  | 6 -> Rotate.rotated_F mrot rastrigin_x
  | 7 -> schwefel_x
  | 8 -> Rotate.rotated_F mrot schwefel_x
  | 9 -> micha_x
  |10 -> Rotate.rotated_F mrot micha_x
  |11 -> helice_x
  |12 -> Rotate.rotated_F mrot helice_x
  | _ -> failwith "Undefined function selector";;

let f_X= 
  match func with
    0 -> expri
  | 1 -> griewank_X
  | 2 -> Rotate.rotated_I mrot griewank_X
  | 3 -> ackley_X
  | 4 -> Rotate.rotated_I mrot ackley_X
  | 5 -> rastrigin_X
  | 6 -> Rotate.rotated_I mrot rastrigin_X
  | 7 -> schwefel_X
  | 8 -> Rotate.rotated_I mrot schwefel_X
  | 9 -> micha_X
  |10 -> Rotate.rotated_I mrot micha_X
  |11 -> helice_X
  |12 -> Rotate.rotated_I mrot helice_X
  | _ -> failwith "Undefined function selector";;

