/*
    Copyright 2011 Jean-Marc Alliot

    This file is part of the mpfi/mpfr bindings.

    The mpfi/mpfr bindings is free software: 
    you can redistribute it and/or modify it under the terms of 
    the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The mpfi/mpfr bindings is distributed in the hope that it will be 
    useful,but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public 
    License along with the mpfi/mpfr bindings.  
    If not, see <http://www.gnu.org/licenses/>.
*/



#include <stdio.h>
#include <mpfr.h>


#include <caml/mlvalues.h>
#include <caml/memory.h>
#include <caml/alloc.h>
#include <caml/custom.h>

/* Encapsulation of opaque mpfr handles (of type mpfr_t *)
   as Caml custom blocks. */

void finalize_mpfr(value v);
int compare_mpfr(value vm1,value vm2);

static struct custom_operations mpfr_ops = {
  "alliot.fr.mpfr",
  finalize_mpfr,
  compare_mpfr,
  custom_hash_default,
  custom_serialize_default,
  custom_deserialize_default
};

/* Accessing the mpfr_t * part of a Caml custom block */
#define Mpfr_val(v) (*((mpfr_t **) Data_custom_val(v)))

void finalize_mpfr(value v)
{
  mpfr_t *m=Mpfr_val(v);
  mpfr_clear(*m);
  free(m);
}

int compare_mpfr(value vm1,value vm2)
{
  mpfr_t *m1=Mpfr_val(vm1);
  mpfr_t *m2=Mpfr_val(vm2);
  return Int_val(mpfr_cmp(*m1,*m2));
}

value caml_mpfr_init()
{
  CAMLlocal1(v);
  mpfr_t *m;
  m=(mpfr_t *)malloc(sizeof(mpfr_t));
  mpfr_init(*m);
  v = alloc_custom(&mpfr_ops, sizeof(mpfr_t *), 0, 1);
  Mpfr_val(v) = m;
  return v;
}

value caml_mpfr_init2(value vd)
{
  CAMLlocal1(v);
  mpfr_t *m;
  m=(mpfr_t *)malloc(sizeof(mpfr_t));
  mpfr_init2(*m,Val_int(vd));
  v = alloc_custom(&mpfr_ops, sizeof(mpfr_t *), 0, 1);
  Mpfr_val(v) = m;
  return v;
}

void caml_mpfr_set_default_prec(value vi)
{
  mpfr_set_default_prec(Int_val(vi));
}

value caml_mpfr_get_default_prec()
{
  return Val_int (mpfr_get_default_prec());
}

void caml_mpfr_set_prec(value vm,value vi)
{
  mpfr_set_prec(*(Mpfr_val(vm)),Int_val(vi));
}

value caml_mpfr_get_prec(value vm)
{
  return Val_int (mpfr_get_prec(*(Mpfr_val(vm))));
}

value caml_mpfr_set(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_set(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_set_d(value vm,value vd,value vi)
{
  return Val_int(mpfr_set_d(*(Mpfr_val(vm)),Double_val(vd),Int_val(vi)));;
}

value caml_mpfr_set_str(value vm,value vs,value vd,value vi)
{
  return Val_int(mpfr_set_str(*(Mpfr_val(vm)),String_val(vs),Int_val(vd),Int_val(vi)));
}

value caml_mpfr_add(value vm1,value vm2,value vm3,value vi)
{
  return Val_int(mpfr_add(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),*(Mpfr_val(vm3)),Int_val(vi)));
}

value caml_mpfr_sub(value vm1,value vm2,value vm3,value vi)
{
  return Val_int(mpfr_sub(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),*(Mpfr_val(vm3)),Int_val(vi)));
}

value caml_mpfr_mul(value vm1,value vm2,value vm3,value vi)
{
  return Val_int(mpfr_mul(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),*(Mpfr_val(vm3)),Int_val(vi)));
}

value caml_mpfr_div(value vm1,value vm2,value vm3,value vi)
{
  return Val_int(mpfr_div(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),*(Mpfr_val(vm3)),Int_val(vi)));
}

value caml_mpfr_neg(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_neg(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_sqr(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_sqr(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_sqrt(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_sqrt(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_abs(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_abs(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_log(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_log(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_exp(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_exp(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_exp2(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_exp2(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_cos(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_cos(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_sin(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_sin(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_tan(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_tan(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}


value caml_mpfr_acos(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_acos(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_asin(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_asin(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_atan(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_atan(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_acosh(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_acosh(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_asinh(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_asinh(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_atanh(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_atanh(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_cosh(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_cosh(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_sinh(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_sinh(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_tanh(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_tanh(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_log1p(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_log1p(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_expm1(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_expm1(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_log2(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_log2(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_log10(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_log10(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_add_d(value vm1,value vm2,value vd,value vi)
{
  return Val_int(mpfr_add_d(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Double_val(vd),Int_val(vi)));
}

value caml_mpfr_sub_d(value vm1,value vm2,value vd,value vi)
{
  return Val_int(mpfr_sub_d(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Double_val(vd),Int_val(vi)));
}

value caml_mpfr_mul_d(value vm1,value vm2,value vd,value vi)
{
  return Val_int(mpfr_mul_d(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Double_val(vd),Int_val(vi)));
}

value caml_mpfr_div_d(value vm1,value vm2,value vd,value vi)
{
  return Val_int(mpfr_div_d(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Double_val(vd),Int_val(vi)));
}

value caml_mpfr_d_sub(value vm1,value vd,value vm2,value vi)
{
  return Val_int(mpfr_d_sub(*(Mpfr_val(vm1)),Double_val(vd),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_d_div(value vm1,value vd,value vm2,value vi)
{
  return Val_int(mpfr_d_div(*(Mpfr_val(vm1)),Double_val(vd),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_get_d(value vm1,value vi)
{
  return caml_copy_double(mpfr_get_d(*(Mpfr_val(vm1)),Int_val(vi)));
}

value caml_mpfr_const_log2(value vm1,value vi)
{
  return Val_int(mpfr_const_log2(*(Mpfr_val(vm1)),Int_val(vi)));
}

value caml_mpfr_const_pi(value vm1,value vi)
{
  return Val_int(mpfr_const_pi(*(Mpfr_val(vm1)),Int_val(vi)));
}

value caml_mpfr_const_euler(value vm1,value vi)
{
  return Val_int(mpfr_const_euler(*(Mpfr_val(vm1)),Int_val(vi)));
}

value caml_mpfr_cmp(value vm1,value vm2)
{
  return Val_int(mpfr_cmp(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}

value caml_mpfr_cmp_d(value vm1,value vd)
{
  return Val_int(mpfr_cmp_d(*(Mpfr_val(vm1)),Double_val(vd)));
}

value caml_mpfr_cmp_abs(value vm1,value vm2)
{
  return Val_int(mpfr_cmp_abs(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}

value caml_mpfr_nan_p(value vm1)
{
  return Val_int(mpfr_nan_p(*(Mpfr_val(vm1))));
}

value caml_mpfr_inf_p(value vm1)
{
  return Val_int(mpfr_inf_p(*(Mpfr_val(vm1))));
}

value caml_mpfr_number_p(value vm1)
{
  return Val_int(mpfr_number_p(*(Mpfr_val(vm1))));
}

value caml_mpfr_zero_p(value vm1)
{
  return Val_int(mpfr_zero_p(*(Mpfr_val(vm1))));
}

value caml_mpfr_regular_p(value vm1)
{
  return Val_int(mpfr_regular_p(*(Mpfr_val(vm1))));
}

value caml_mpfr_sgn(value vm1)
{
  return Val_int(mpfr_sgn(*(Mpfr_val(vm1))));
}

value caml_mpfr_greater_p(value vm1,value vm2)
{
  return Val_int(mpfr_greater_p(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}

value caml_mpfr_greaterequal_p(value vm1,value vm2)
{
  return Val_int(mpfr_greaterequal_p(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}

value caml_mpfr_less_p(value vm1,value vm2)
{
  return Val_int(mpfr_less_p(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}

value caml_mpfr_lessequal_p(value vm1,value vm2)
{
  return Val_int(mpfr_lessequal_p(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}

value caml_mpfr_equal_p(value vm1,value vm2)
{
  return Val_int(mpfr_equal_p(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}

value caml_mpfr_lessgreater_p(value vm1,value vm2)
{
  return Val_int(mpfr_lessgreater_p(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}

value caml_mpfr_unordered_p(value vm1,value vm2)
{
  return Val_int(mpfr_unordered_p(*(Mpfr_val(vm1)),*(Mpfr_val(vm2))));
}


value caml_mpfr_out_str(value vi1,value vi2,value vm,value vi3)
{
  return Val_int(mpfr_out_str(stdout,Int_val(vi1),Int_val(vi2),*(Mpfr_val(vm)),Val_int(vi3)));
}




value caml_mpfr_cbrt(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_cbrt(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_csc(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_csc(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_sec(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_sec(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_cot(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_cot(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_csch(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_csch(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_sech(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_sech(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_coth(value vm1,value vm2,value vi)
{
  return Val_int(mpfr_coth(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),Int_val(vi)));
}

value caml_mpfr_atan2(value vm1,value vm2,value vm3,value vi)
{
  return Val_int(mpfr_atan2(*(Mpfr_val(vm1)),*(Mpfr_val(vm2)),*(Mpfr_val(vm3)),Int_val(vi)));
}

value caml_mpfr_const_catalan(value vm1,value vi)
{
  return Val_int(mpfr_const_catalan(*(Mpfr_val(vm1)),Int_val(vi)));
}

